#include "BruteForceSudokuSolver.h"
#include <iostream>
#include <string>
#include <fstream>
#include <cctype>  // For isdigit()
#include <cstdlib> // For atoi()
using std::string;
using std::ifstream;
using std::ofstream;
using std::cout;
using std::cerr;
using std::endl;

// Reads a Sudoku puzzle file and populates the puzzle values in a BruteForceSudokuSolver object.
//
// Parameters:
//  pFilename: The name of a file containing a Sudoku puzzle in text format
//  pSolver: A reference to a BruteForceSudokuSolver object to contain the puzzle values
//
// Return value: Boolean - true if successful, or false if not
bool readPuzzleFile(const char* pFilename, BruteForceSudokuSolver& pSolver);

int main(int argc, char* argv[])
{
	int retval = 0;

	if (argc > 1)
	{
		BruteForceSudokuSolver solver;
		if (readPuzzleFile(argv[1], solver))
		{
			if (solver.numGivenNumbers() >= 17)
			{
				cout << "Solving..." << endl;
				solver.solve();
				// If a solution was found, then output the solution to the
				// file, with the same filename as given on the command line.
				if (solver.NumSolutions() > 0)
				{
					ofstream outFile(argv[1], std::ios_base::out|std::ios_base::trunc);
					if (outFile.is_open())
					{
						// Write the solution to the file
						for (int row = 0; row < 9; ++row)
						{
							for (int col = 0; col < 9; ++col)
								outFile << solver.getSolvedVal(0, row, col);
							outFile << endl;
						}
						outFile.close();
						cout << "Done solving." << endl;
					}
					else
					{
						retval = 1;
						cerr << "Error: Could not write to " << argv[1] << endl;
					}
				}
			}
			else
			{
				retval = 2;
				cerr << "Error: The puzzle must have at least 17 given numbers." << endl;
			}
		}
		else
		{
			retval = 3;
			cerr << "Error reading " << argv[1] << endl;
		}
	}
	else
	{
		retval = 4;
		cerr << "Error: No filename specified." << endl;
	}

	return retval;
}

bool readPuzzleFile(const char* pFilename, BruteForceSudokuSolver& pSolver)
{
	if (pFilename == NULL)
		return false;

	bool retval = false;

	ifstream inFile(pFilename, std::ios_base::in);
	if (inFile.is_open())
	{
		retval = true;
		string fileLine;      // A text line read from the file
		char fileChar = ' ';  // A single character read from the file
		int valInt = 0;       // A value from the file, converted to an int
		int row = 0;
		int col = 0;
		size_t lineLen = 0;
		while (inFile.good() && (row < 9))
		{
			std::getline(inFile, fileLine);
			lineLen = fileLine.length();
			// Skip blank lines and lines starting with a #
			if (lineLen == 0)
				continue;
			if (fileLine[0] == '#')
				continue;

			// Verify that the line has at least 9 characters.  If not, then
			// stop reading the file.
			if (lineLen < 9)
			{
				cerr << "Invalid puzzle: Line " << row+1 << " has less than 9 characters." << endl;
				retval = false;
				break;
			}
			// Verify that the first 9 characters are digits or spaces.
			for (col = 0; col < 9; ++col)
			{
				// Treat a space or a . as a zero.
				if ((fileLine[col] == ' ') || (fileLine[col] == '.'))
					pSolver.setVal(row, col, 0);
				else if (isdigit(fileLine[col]))
				{
					// Convert the character to a digit and set it in the solver.
					fileChar = fileLine[col];
					valInt = atoi(&fileChar);
					pSolver.setVal(row, col, valInt);
				}
				else
				{
					cerr << "Invalid puzzle: Line" << row+1 << " has a non-digit or non-space." << endl;
					retval = false;
					break;
				}
			}

			++row;
		}
		inFile.close();

		// If we didn't read 9 rows from the file, then assume the puzzle is not valid.
		if (row < 9)
			retval = false;
	}

	return retval;
}